"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserializeSignature = exports.serializeSignatureInfo = void 0;
/*
 * This public interface for serializing the AWS Encryption SDK Message Footer Format
 * is provided for the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other than the
 * Encryption SDK for JavaScript.
 *
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#footer-structure
 */
const concat_buffers_1 = require("./concat_buffers");
const uint_util_1 = require("./uint_util");
const material_management_1 = require("@aws-crypto/material-management");
function serializeSignatureInfo(signature) {
    return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt16BE)(signature.byteLength), signature);
}
exports.serializeSignatureInfo = serializeSignatureInfo;
function deserializeSignature({ buffer, byteOffset, byteLength, }) {
    /* Precondition: There must be information for a signature. */
    (0, material_management_1.needs)(byteLength && byteLength > 2, 'Invalid Signature');
    /* Uint8Array is a view on top of the underlying ArrayBuffer.
     * This means that raw underlying memory stored in the ArrayBuffer
     * may be larger than the Uint8Array.  This is especially true of
     * the Node.js Buffer object.  The offset and length *must* be
     * passed to the DataView otherwise I will get unexpected results.
     */
    const dataView = new DataView(buffer, byteOffset, byteLength);
    const signatureLength = dataView.getUint16(0, false); // big endian
    /* Precondition: The signature length must be positive. */
    (0, material_management_1.needs)(signatureLength > 0, 'Invalid Signature');
    /* Precondition: The data must match the serialized length. */
    (0, material_management_1.needs)(byteLength === signatureLength + 2, 'Invalid Signature');
    return new Uint8Array(buffer, byteOffset + 2, signatureLength);
}
exports.deserializeSignature = deserializeSignature;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2lnbmF0dXJlX2luZm8uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvc2lnbmF0dXJlX2luZm8udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7OztBQUV0Qzs7Ozs7OztHQU9HO0FBRUgscURBQWdEO0FBQ2hELDJDQUFzQztBQUN0Qyx5RUFBdUQ7QUFFdkQsU0FBZ0Isc0JBQXNCLENBQUMsU0FBcUI7SUFDMUQsT0FBTyxJQUFBLDhCQUFhLEVBQUMsSUFBQSxvQkFBUSxFQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBQTtBQUNqRSxDQUFDO0FBRkQsd0RBRUM7QUFFRCxTQUFnQixvQkFBb0IsQ0FBQyxFQUNuQyxNQUFNLEVBQ04sVUFBVSxFQUNWLFVBQVUsR0FDQztJQUNYLDhEQUE4RDtJQUM5RCxJQUFBLDJCQUFLLEVBQUMsVUFBVSxJQUFJLFVBQVUsR0FBRyxDQUFDLEVBQUUsbUJBQW1CLENBQUMsQ0FBQTtJQUN4RDs7Ozs7T0FLRztJQUNILE1BQU0sUUFBUSxHQUFHLElBQUksUUFBUSxDQUFDLE1BQU0sRUFBRSxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUE7SUFDN0QsTUFBTSxlQUFlLEdBQUcsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUEsQ0FBQyxhQUFhO0lBQ2xFLDBEQUEwRDtJQUMxRCxJQUFBLDJCQUFLLEVBQUMsZUFBZSxHQUFHLENBQUMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFBO0lBQy9DLDhEQUE4RDtJQUM5RCxJQUFBLDJCQUFLLEVBQUMsVUFBVSxLQUFLLGVBQWUsR0FBRyxDQUFDLEVBQUUsbUJBQW1CLENBQUMsQ0FBQTtJQUM5RCxPQUFPLElBQUksVUFBVSxDQUFDLE1BQU0sRUFBRSxVQUFVLEdBQUcsQ0FBQyxFQUFFLGVBQWUsQ0FBQyxDQUFBO0FBQ2hFLENBQUM7QUFwQkQsb0RBb0JDIn0=