# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import argparse

from tools.common.validators.input_validators import FileValidator, PathNameValidator, StringValidator


class ValidFileSpec:
    """
    ArgumentParser "type" validator for command line options that accept file names/paths
    """

    def __init__(self, file_must_exist=True, file_must_not_exist=False, max_file_size=0, allow_symlinks=True):
        """
        Constructor

        :param file_must_exist: when True, verifies that the specified file exists
        :param file_must_not_exist: when True, verifies that the specified file does not exist
        :param max_file_size: maximum file size allowed (0 means MAX_FILE_SIZE_BYTES)
        :param allow_symlinks: when True, allow using symlinks for existing files, otherwise reject symlinks
        """
        self.__validate = FileValidator(file_must_exist=file_must_exist,
                                        file_must_not_exist=file_must_not_exist,
                                        max_file_size=max_file_size,
                                        allow_symlinks=allow_symlinks
                                        )

    def __call__(self, file_spec):
        try:
            return self.__validate(file_spec)
        except ValueError as e:
            raise argparse.ArgumentTypeError(e)


class ValidString:

    def __init__(self):
        self.__validate = StringValidator()

    def __call__(self, string):
        try:
            return self.__validate(string)
        except ValueError as e:
            raise argparse.ArgumentTypeError(e)

class ValidPathName:

    def __init__(self):
        self.__validate = PathNameValidator()

    def __call__(self, string):
        try:
            return self.__validate(string)
        except ValueError as e:
            raise argparse.ArgumentTypeError(e)


class ValidPathNameAndFileSpec:

    def __init__(self, file_must_exist=True, file_must_not_exist=False, max_file_size=0, allow_symlinks=True):
        self.path_name_validator = ValidPathName()
        self.file_spec_validator = ValidFileSpec(file_must_exist, file_must_not_exist, max_file_size, allow_symlinks)

    def __call__(self, string):
        try:
            self.path_name_validator(string)
            return self.file_spec_validator(string)
        except ValueError as e:
            raise argparse.ArgumentTypeError(e)

